/*
  ==============================================================================

    SonicCrypt Seq
    Copyright (C) 2025 Sebastian Sünkler

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

  ==============================================================================
*/

#pragma once
#include <JuceHeader.h>
#include "SequencerData.h"

struct ActiveNote {
    int noteNumber;
    int samplesRemaining;
};

struct LayerRuntime {
    int currentStep = 0;
    int currentOctave = 0;
    int samplesProcessed = 0;
    bool isMovingForward = true;
};

class NewProjectAudioProcessor : public juce::AudioProcessor
{
public:
    NewProjectAudioProcessor();
    ~NewProjectAudioProcessor() override;

    void prepareToPlay(double sampleRate, int samplesPerBlock) override;
    void releaseResources() override;
    bool isBusesLayoutSupported(const BusesLayout& layouts) const override;
    void processBlock(juce::AudioBuffer<float>&, juce::MidiBuffer&) override;

    juce::AudioProcessorEditor* createEditor() override;
    bool hasEditor() const override;

    const juce::String getName() const override;
    bool acceptsMidi() const override;
    bool producesMidi() const override;
    bool isMidiEffect() const override;
    double getTailLengthSeconds() const override;

    int getNumPrograms() override;
    int getCurrentProgram() override;
    void setCurrentProgram(int index) override;
    const juce::String getProgramName(int index) override;
    void changeProgramName(int index, const juce::String& newName) override;

    void getStateInformation(juce::MemoryBlock& destData) override;
    void setStateInformation(const void* data, int sizeInBytes) override;

    juce::File getPresetFolder() const;
    void savePreset(const juce::String& name);
    void loadPresetFile(const juce::File& file);

    SequencerData& getSequencerData() { return mySequencer; }

    int getCurrentStepForLayer(int layerIndex) const {
        if (layerIndex >= 0 && layerIndex < 3) return layerRuntime[layerIndex].currentStep;
        return 0;
    }

    // Export Function
    juce::File createMidiFileForExport(int numBars = 4);

    // --- UI SIZE PERSISTENCE ---
    int lastUIWidth = 1150;
    int lastUIHeight = 650;

private:
    SequencerData mySequencer;
    std::array<LayerRuntime, 3> layerRuntime;
    int activeInputNote = -1;
    std::vector<ActiveNote> activeNotes;

    JUCE_DECLARE_NON_COPYABLE_WITH_LEAK_DETECTOR(NewProjectAudioProcessor)
};